/* This file is part of OpenBubbles.
 *
 * OpenBubbles is an SDL clone of Bubbles.
 * Copyright (C) 2004  Benny Sperisen
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * button.cpp is the implementation file of button.h.
 *
 * Written by:
 *  Benny Sperisen
 *  lasindi@gmail.com
 *  www.freewebs.com/lasindi
 *****************************************************************************/

#include "button.h"

button::button(int x,int y,string image,char* name,
							 string image_hilight,string image_pressed)
:_x(x),_y(y),_name(name),_state(NORMAL)
{
	string temp=image,temp_h=image_hilight,temp_p=image_pressed;
	image=DATA_PREFIX;
	image+="/"+temp;
  _image=IMG_Load(image.c_str());
  if(_image==NULL)
  {
    cerr<<"ERROR: unable to load "<<image<<".\n";
    exit(1);
  }
  _w=_image->w;
  _h=_image->h;
  // Load the (optional) hilight and pressed images.
  if(image_hilight!="")
  {
  	image_hilight=DATA_PREFIX;
	  image_hilight+="/"+temp_h;
    _image_hilight=IMG_Load(image_hilight.c_str());
    if(_image_hilight==NULL)
    {
      cerr<<"ERROR: unable to load "<<image_hilight<<".\n";
      exit(1);
    }
  }
  else
    _image_hilight=NULL;
  if(image_pressed!="")
  {
  	image_pressed=DATA_PREFIX;
	  image_pressed+="/"+temp_p;
    _image_pressed=IMG_Load(image_pressed.c_str());
    if(_image_pressed==NULL)
    {
      cerr<<"ERROR: unable to load "<<image_pressed<<".\n";
      exit(1);
    }
  }
  else
    _image_hilight=NULL;
}

void button::draw()
{
  SDL_Rect src,dest;
  src.x=0;
  src.y=0;
  dest.x=_x;
  dest.y=_y;
  switch(_state)
  {
  case NORMAL:
    src.w=_image->w;
    src.h=_image->h;
    dest.w=src.w;
    dest.h=src.h;
    SDL_BlitSurface(_image,&src,screen,&dest);
    break;
  case HILIGHT:
    src.w=_image_hilight->w;
    src.h=_image_hilight->h;
    dest.w=src.w;
    dest.h=src.h;
    SDL_BlitSurface(_image_hilight,&src,screen,&dest);
    break;
  default:
    src.w=_image_pressed->w;
    src.h=_image_pressed->h;
    dest.w=src.w;
    dest.h=src.h;
    SDL_BlitSurface(_image_pressed,&src,screen,&dest);
  }
}

bool button::within(int x,int y)
{
  if((x>=_x)&&(y>=_y)&&(x<_x+_w)&&(y<_y+_h))
    return true;
  return false;
}

char button::state() const
{
  return _state;
}

char* button::name() const
{
  return _name;
}

void button::setState(char state)
{
  switch(state)
  {
  case NORMAL:
    _state=state;
    break;
  case HILIGHT:
    if(_image_hilight!=NULL)
      _state=state;
    break;
  case PRESSED:
    if(_image_pressed!=NULL)
      _state=state;
    break;
  default:
    cerr<<"ERROR: tried to set illegal state in button "<<_name<<endl;
    exit(1);
  }
}

void button::freeSurfaces()
{
  SDL_FreeSurface(_image);
  if(_image_hilight!=NULL)
    SDL_FreeSurface(_image_hilight);
  if(_image_pressed!=NULL)
    SDL_FreeSurface(_image_pressed);
}

checkbox::checkbox(int x,int y,char *name,bool check)
	:button(x,y,"check.png",name,"check_hilight.png",
					"check_pressed.png")
{
	string imagedir=DATA_PREFIX;
	_checked=check;
	imagedir+="/checkbox.png";
  _image_unchecked=IMG_Load(imagedir.c_str());
  if(_image_unchecked==NULL)
  {
    cerr<<"ERROR: unable to load "<<imagedir<<".\n";
    exit(1);
  }
	imagedir=DATA_PREFIX;
	imagedir+="/checkbox_hilight.png";
	_unchecked_hilight=IMG_Load(imagedir.c_str());
	if(_unchecked_hilight==NULL)
	{
		cerr<<"ERROR: unable to load "<<imagedir<<".\n";
		exit(1);
	}
}

void checkbox::draw()
{
  SDL_Rect src,dest;
  src.x=0;
  src.y=0;
  dest.x=_x;
  dest.y=_y;
  switch(_state)
  {
  case NORMAL:
    src.w=_image->w;
    src.h=_image->h;
    dest.w=src.w;
    dest.h=src.h;
		if(_checked)
      SDL_BlitSurface(_image,&src,screen,&dest);
		else
			SDL_BlitSurface(_image_unchecked,&src,screen,&dest);
    break;
  case HILIGHT:
    src.w=_image_hilight->w;
    src.h=_image_hilight->h;
    dest.w=src.w;
    dest.h=src.h;
		if(_checked)
      SDL_BlitSurface(_image_hilight,&src,screen,&dest);
		else
			SDL_BlitSurface(_unchecked_hilight,&src,screen,&dest);
    break;
  default:
    src.w=_image_pressed->w;
    src.h=_image_pressed->h;
    dest.w=src.w;
    dest.h=src.h;
    SDL_BlitSurface(_image_pressed,&src,screen,&dest);
  }
}

void checkbox::setState(char state)
{
  switch(state)
  {
  case NORMAL:
    _state=state;
    break;
  case HILIGHT:
    _state=state;
    break;
  case PRESSED:
    _state=state;
 		_checked=!_checked;
		break;
  default:
    cerr<<"ERROR: tried to set illegal state in button "<<_name<<endl;
    exit(1);
  }
}

bool checkbox::checked() const
{
	return _checked;
}

void checkbox::freeSurfaces()
{
  SDL_FreeSurface(_image);
  SDL_FreeSurface(_image_hilight);
  SDL_FreeSurface(_image_pressed);
	SDL_FreeSurface(_image_unchecked);
	SDL_FreeSurface(_unchecked_hilight);
}

list<button*> buttons;
list<checkbox*> checkboxes;
char prevclickstate=0;

char* handleButtons()
{
  Uint8 newclickstate;
  int x,y;
  SDL_PumpEvents();
  newclickstate=SDL_GetMouseState(&x,&y)&SDL_BUTTON(1);
  // The user either pressed or released the left mouse button.
  if(newclickstate!=prevclickstate)
  {
    if(newclickstate) // The user pressed the buttons.
		{
      for(list<button*>::iterator i=buttons.begin();i!=buttons.end();i++)
        if((*i)->within(x,y))
          (*i)->setState(PRESSED);
		}
    else // The user released the button.
		{
      for(list<button*>::iterator i=buttons.begin();i!=buttons.end();i++)
        // The user did a "full" click on the button, so return its name.
        if(((*i)->within(x,y))&&((*i)->state()==PRESSED))
        {
          (*i)->setState(HILIGHT);
          return (*i)->name();
        }
		}
    prevclickstate=newclickstate;
  }
  else
    for(list<button*>::iterator i=buttons.begin();i!=buttons.end();i++)
			// The mouse is over the button, so highlight it.
      if(((*i)->within(x,y))&&((*i)->state()!=PRESSED)) 
        (*i)->setState(HILIGHT);
      else if(!(*i)->within(x,y)) // The mouse moved off of the button, so make
        (*i)->setState(NORMAL);   // it normal.
  return "";
}

void drawButtons()
{
  for(list<button*>::iterator i=buttons.begin();i!=buttons.end();i++)
    (*i)->draw();
}

bool isChecked(string name)
{
	for(list<checkbox*>::iterator it=checkboxes.begin();it!=checkboxes.end();
			it++)
	{
		if((*it)->name()==name)
			if((*it)->checked())
				return true;
			else
				return false;
	}
	return false;
}
