#
# Install local dependency from external deb package
#
# Functions:
#
#   target_architecture_deb (output_var)
#       Sets output_var to current architecture (usable for Debin repos)
#        output_var     name of return variable
#
#   ExternalDeb_add (NAME URL)
#       Downloads and installs external deb from URL.
#
#       Parameters:
#        NAME   name of deb (used for internal naming)
#        URL    full URL to deb, with %ARCH% variable, which is replaced with current arch
#               (basing on target_architecture_deb function) – should be usable for wget
#
#       Returns:
#        ${NAME}_FOUND          if deb was installed properly
#        ${NAME}_INCLUDE_DIRS   list of include dirs
#        ${NAME}_LIBRARY_DIRS   list of library dirs (included in link_directories list)
#        ${NAME}_SLIBRARIES     list of static libraries
#        ${NAME}_DLIBRARIES     list of dynamic libraries
#           All lists are generated by this simple cmake script, so they may be incomplete!
#
#       Dependencies:
#        wget   for downloading deb from URL
#        ar     to unpack deb archive
#        tar    to unpack internal data archive
#
#
# Copyright (c) 2015, Marek Pikuła <marek@pikula.co>
# All rights reserved.
#
# Distributed under the OSI-approved BSD License (the "License") see accompanying file Copyright.txt for details.
#
# This software is distributed WITHOUT ANY WARRANTY without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE. See the License for more information.
#

function (target_architecture_deb output_var)
    include (TargetArch)
    target_architecture(ARCH)

    if (${ARCH} STREQUAL armv7)
        set (DARCH armhf)
    elseif (${ARCH} STREQUAL armv6)
        set (DARCH armel)
    elseif (${ARCH} STREQUAL armv5)
        set (DARCH armel)
    elseif (${ARCH} STREQUAL arm)
        set (DARCH arm)
    elseif (${ARCH} STREQUAL i386)
        set (DARCH i386)
    elseif (${ARCH} STREQUAL x86_64)
        set (DARCH amd64)
    elseif (${ARCH} STREQUAL ia64)
        set (DARCH ia64)
    elseif (${ARCH} STREQUAL ppc64)
        set (DARCH ppc64)
    elseif (${ARCH} STREQUAL ppc)
        set (DARCH powerpc)
    elseif (${ARCH} STREQUAL unknown)
        set (DARCH unknown)
    endif (${ARCH} STREQUAL armv7)

    set (${output_var} ${DARCH} PARENT_SCOPE)
endfunction (target_architecture_deb)

function (ExternalDeb_add NAME URL)

    find_program (PATH_WGET wget)
    if (NOT PATH_WGET)
        message (FATAL_ERROR "wget not found (needed to download deb files)")
    endif (NOT PATH_WGET)

    find_program (PATH_AR ar)
    if (NOT PATH_AR)
        message (FATAL_ERROR "ar not found (needed to unpack deb files)")
    endif (NOT PATH_AR)

    find_program (PATH_TAR tar)
    if (NOT PATH_TAR)
        message (FATAL_ERROR "tar not found (needed to unpack deb files)")
    endif (NOT PATH_TAR)

    target_architecture_deb (ARCH)
    set (URL "${URL}")
    string (REPLACE "%ARCH%" ${ARCH} URL "${URL}")

    set (DEB_PREFIX "${CMAKE_BINARY_DIR}/${NAME}-prefix")
    file (MAKE_DIRECTORY "${DEB_PREFIX}")

    if (NOT EXISTS "${DEB_PREFIX}/usr")
        execute_process (COMMAND ${PATH_WGET} -O ${NAME}.deb ${URL} WORKING_DIRECTORY "${DEB_PREFIX}")
        execute_process (COMMAND ${PATH_AR} x ${NAME}.deb WORKING_DIRECTORY "${DEB_PREFIX}")
        execute_process (COMMAND ${PATH_TAR} -xf data.tar.xz WORKING_DIRECTORY "${DEB_PREFIX}")
    endif (NOT EXISTS "${DEB_PREFIX}/usr")

    set (${NAME}_FOUND TRUE PARENT_SCOPE)

    file (GLOB INCLUDE_DIRS "${DEB_PREFIX}/usr/include/*")
    set (${NAME}_INCLUDE_DIRS "${INCLUDE_DIRS}" PARENT_SCOPE)

    file (GLOB LIBRARY_DIRS "${DEB_PREFIX}/usr/lib/*")
    set (${NAME}_LIBRARY_DIRS "${LIBRARY_DIRS}" PARENT_SCOPE)
    link_directories (${LIBRARY_DIRS})

    foreach (DIRECTORY ${LIBRARY_DIRS})
        file (GLOB SLIBRARIES_TMP "${DIRECTORY}/*.a")
        set (${NAME}_SLIBRARIES ${${NAME}_SLIBRARIES} ${SLIBRARIES_TMP} PARENT_SCOPE)

        file (GLOB DLIBRARIES_TMP "${DIRECTORY}/*.so")
        set (${NAME}_DLIBRARIES ${${NAME}_DLIBRARIES} ${DLIBRARIES_TMP} PARENT_SCOPE)
    endforeach (DIRECTORY)

endfunction (ExternalDeb_add)
