#!/usr/bin/perl
#
# Copyright (c) 2018  Peter Pentchev
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.

use v5.010;
use strict;
use warnings;

use Test::More;
use Test::Command;

use lib 't/lib';
use Test::FeatureCheck qw(env_init get_error_output get_ok_output);

my %c = env_init;

my $version_line;
my @usage_lines;

plan tests => 8;

# A single version line with -V
subtest 'Version output with -V' => sub {
	my $c = Test::Command->new(cmd => [$c{prog}, '-V']);
	$c->exit_is_num(0, '-V succeeded');
	$c->stderr_is_eq('', '-V did not output any errors');
	my @lines = split /\n/, $c->stdout_value, -1;
	BAIL_OUT('Unexpected number of lines in the -V output') unless @lines == 2;
	BAIL_OUT('Unexpected -V line') unless $lines[0] =~ /^ feature-check \s \S+ $/x;
	$version_line = $lines[0];
};

# More than one usage line with -h
subtest 'Usage output with -h' => sub {
	my $c = Test::Command->new(cmd => [$c{prog}, '-h']);
	$c->exit_is_num(0, '-h succeeded');
	$c->stderr_is_eq('', '-h did not output any errors');
	my @lines = split /\n/, $c->stdout_value;
	BAIL_OUT('Too few lines in the -h output') unless @lines > 1;
	if (!$c{is_python}) {
		BAIL_OUT('Unexpected -h first line') unless $lines[0] =~ /^ Usage: \s+ feature-check /x;
	} else {
		BAIL_OUT('Unexpected -h first line') unless $lines[0] =~ /^ usage: \s+ $/x;
	}
	@usage_lines = @lines;
};

if (!$c{is_python}) {
	subtest 'Usage and version output with -V -h' => sub {
		my @lines = get_ok_output([$c{prog}, '-V', '-h'], '-V -h');
		is scalar @lines, 1 + @usage_lines, '-V -h output one more line than the usage message';
		is $lines[0], $version_line, '-V -h output the version line first';
		shift @lines;
		is_deeply \@lines, \@usage_lines, '-V -h output the usage message';
	};

	subtest 'Usage and version output with -hV' => sub {
		my @lines = get_ok_output([$c{prog}, '-hV'], '-hV');
		is scalar @lines, 1 + @usage_lines, '-hV output one more line than the usage message';
		is $lines[0], $version_line, '-hV output the version line first';
		shift @lines;
		is_deeply \@lines, \@usage_lines, '-hV output the usage message';
	};
} else {
	subtest 'Usage output with -V -h' => sub {
		my @lines = get_ok_output([$c{prog}, '-V', '-h'], '-V -h');
		is scalar @lines, @usage_lines, '-V -h output one more line than the usage message';
		is_deeply \@lines, \@usage_lines, '-V -h output the usage message';
	};

	subtest 'Usage and version output with -hV' => sub {
		my @lines = get_ok_output([$c{prog}, '-hV'], '-hV');
		is scalar @lines, @usage_lines, '-hV output one more line than the usage message';
		is_deeply \@lines, \@usage_lines, '-hV output the usage message';
	};
}

subtest 'Long-form version' => sub {
	my @lines = get_ok_output([$c{prog}, '--version'], '--version');
	is scalar @lines, 1, '--version output a single line';
	is $lines[0], $version_line, '--version output the version information';
};

subtest 'Long-form usage' => sub {
	my @lines = get_ok_output([$c{prog}, '--help'], '--help');
	ok @lines > 1, '--help output more than one line';
	is_deeply \@lines, \@usage_lines, '--help output the usage information';
};

subtest 'Invalid short option' => sub {
	my @lines = get_error_output([$c{prog}, '-X', 'Makefile'],
	    'invalid short option -X');
	SKIP: {
		skip 'Usage with unknown options is weird with Python', 2 if $c{is_python};
		is scalar @lines, 1 + scalar @usage_lines, '-X output one more line than the usage message';
		shift @lines;
		is_deeply \@lines, \@usage_lines, '-X output the usage message';
	}
};

subtest 'Invalid long option' => sub {
	my @lines = get_error_output([$c{prog}, '--whee', 'Makefile'],
	    'invalid short option --whee');
	SKIP: {
		skip 'Usage with unknown options is weird with Python', 2 if $c{is_python};
		is scalar @lines, 1 + scalar @usage_lines, '--whee output one more line than the usage message';
		shift @lines;
		is_deeply \@lines, \@usage_lines, '--whee output the usage message';
	}
};
