﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/rds/RDSRequest.h>
#include <aws/rds/RDS_EXPORTS.h>

#include <utility>

namespace Aws {
namespace RDS {
namespace Model {

/**
 */
class DescribeDBMajorEngineVersionsRequest : public RDSRequest {
 public:
  AWS_RDS_API DescribeDBMajorEngineVersionsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeDBMajorEngineVersions"; }

  AWS_RDS_API Aws::String SerializePayload() const override;

 protected:
  AWS_RDS_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The database engine to return major version details for.</p> <p>Valid
   * Values:</p> <ul> <li> <p> <code>aurora-mysql</code> </p> </li> <li> <p>
   * <code>aurora-postgresql</code> </p> </li> <li> <p>
   * <code>custom-sqlserver-ee</code> </p> </li> <li> <p>
   * <code>custom-sqlserver-se</code> </p> </li> <li> <p>
   * <code>custom-sqlserver-web</code> </p> </li> <li> <p> <code>db2-ae</code> </p>
   * </li> <li> <p> <code>db2-se</code> </p> </li> <li> <p> <code>mariadb</code> </p>
   * </li> <li> <p> <code>mysql</code> </p> </li> <li> <p> <code>oracle-ee</code>
   * </p> </li> <li> <p> <code>oracle-ee-cdb</code> </p> </li> <li> <p>
   * <code>oracle-se2</code> </p> </li> <li> <p> <code>oracle-se2-cdb</code> </p>
   * </li> <li> <p> <code>postgres</code> </p> </li> <li> <p>
   * <code>sqlserver-ee</code> </p> </li> <li> <p> <code>sqlserver-se</code> </p>
   * </li> <li> <p> <code>sqlserver-ex</code> </p> </li> <li> <p>
   * <code>sqlserver-web</code> </p> </li> </ul>
   */
  inline const Aws::String& GetEngine() const { return m_engine; }
  inline bool EngineHasBeenSet() const { return m_engineHasBeenSet; }
  template <typename EngineT = Aws::String>
  void SetEngine(EngineT&& value) {
    m_engineHasBeenSet = true;
    m_engine = std::forward<EngineT>(value);
  }
  template <typename EngineT = Aws::String>
  DescribeDBMajorEngineVersionsRequest& WithEngine(EngineT&& value) {
    SetEngine(std::forward<EngineT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A specific database major engine version to return details for.</p>
   * <p>Example: <code>8.4</code> </p>
   */
  inline const Aws::String& GetMajorEngineVersion() const { return m_majorEngineVersion; }
  inline bool MajorEngineVersionHasBeenSet() const { return m_majorEngineVersionHasBeenSet; }
  template <typename MajorEngineVersionT = Aws::String>
  void SetMajorEngineVersion(MajorEngineVersionT&& value) {
    m_majorEngineVersionHasBeenSet = true;
    m_majorEngineVersion = std::forward<MajorEngineVersionT>(value);
  }
  template <typename MajorEngineVersionT = Aws::String>
  DescribeDBMajorEngineVersionsRequest& WithMajorEngineVersion(MajorEngineVersionT&& value) {
    SetMajorEngineVersion(std::forward<MajorEngineVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional pagination token provided by a previous request. If this
   * parameter is specified, the response includes only records beyond the marker, up
   * to the value specified by <code>MaxRecords</code>.</p>
   */
  inline const Aws::String& GetMarker() const { return m_marker; }
  inline bool MarkerHasBeenSet() const { return m_markerHasBeenSet; }
  template <typename MarkerT = Aws::String>
  void SetMarker(MarkerT&& value) {
    m_markerHasBeenSet = true;
    m_marker = std::forward<MarkerT>(value);
  }
  template <typename MarkerT = Aws::String>
  DescribeDBMajorEngineVersionsRequest& WithMarker(MarkerT&& value) {
    SetMarker(std::forward<MarkerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The maximum number of records to include in the response. If more than the
   * <code>MaxRecords</code> value is available, a pagination token called a marker
   * is included in the response so you can retrieve the remaining results.</p>
   * <p>Default: 100</p>
   */
  inline int GetMaxRecords() const { return m_maxRecords; }
  inline bool MaxRecordsHasBeenSet() const { return m_maxRecordsHasBeenSet; }
  inline void SetMaxRecords(int value) {
    m_maxRecordsHasBeenSet = true;
    m_maxRecords = value;
  }
  inline DescribeDBMajorEngineVersionsRequest& WithMaxRecords(int value) {
    SetMaxRecords(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_engine;
  bool m_engineHasBeenSet = false;

  Aws::String m_majorEngineVersion;
  bool m_majorEngineVersionHasBeenSet = false;

  Aws::String m_marker;
  bool m_markerHasBeenSet = false;

  int m_maxRecords{0};
  bool m_maxRecordsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RDS
}  // namespace Aws
